/** @file   npccontroller.cpp
 * @brief   Implementation of NpcController - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "npccontroller.h"
#include "npccharacter.h"
#include "utils.h"
#include "warglobals.h"
#include "npccharacter.h"

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
NpcController::NpcController( GameObject* pO ) :
  WaypointController( pO ),
  m_npcState( NPC_FREEZE ),
  m_pFollowObject( 0 ),
  m_followDistance( 50 ),
  m_shouldFollow( false ),
  m_aliensInSight( false ),
  m_scanDelay( rand() % 40 ),
  m_pAlienToShoot( 0 )
{
}



/** Destructor
 */
NpcController::~NpcController()
{
}




///
/// Public methods
/// ==============

/** THe update method
 */
void NpcController::update()
{
  // Scan for the aliens
  m_scanDelay -= 1;
  if ( m_scanDelay < 0 ) {
    m_scanDelay = 10 + rand() % 40;
    m_aliensInSight = this->scanForAliens();
  }
  
  switch ( this->state() ) {
    case ( NPC_FREEZE ): {
      this->reset();
      this->m_counter = 5 + rand() % 10;
      break;
    }
    case ( NPC_FOLLOW_OBJECT ): {
      this->followObject();
      break;
    }
    case ( NPC_RANDOM_MOVEMENT ): {
      this->reset();
      m_forward = 1;
      m_turn = (rand() % 3) - 1;
      m_counter = 5 + rand() % 10;
      break;
    }
    case ( NPC_ROUTE ): {
      this->followRoute();
      break;
    }
    case ( NPC_FIGHT ): {
      this->fight();
      break;
    }
  }
  this->updateNpc();
}



/** Sets the state of this controller.
 */
void NpcController::state( NpcState nstate )
{
  m_npcState = nstate;
}



/** Sets the distance how close to the leader we're trying to stay
 */
void NpcController::setFollowDistance( float followDistance )
{
  m_followDistance = followDistance;
}




///
/// Getter methods
/// ==============

/** Returns the state this controller is in.
 */
NpcController::NpcState NpcController::state() const
{
  return m_npcState;
}



///
/// Private or Protected methods
/// ============================

/** Makes our object to follow another object.
 */
void NpcController::followObject()
{
  // If there are aliens in sight and we have weapon, we start shooting.
  NpcCharacter* pObj = dynamic_cast<NpcCharacter*>( m_pObject );
  Weapon tmpWeapon = pObj->getWeapon();
  if ( m_aliensInSight == true && tmpWeapon.id() != Weapon::W_NOWEAPON ) {
    this->fight();
    return;
  }
  
  if ( m_pFollowObject == 0 ) {
    this->reset();
    this->m_counter = 5 + rand() % 10;
    return;
  }
  
      
  // We follow the object only, if we can see him
  bool cansee = Utils::objectsSeeEachOther( this->m_pObject, m_pFollowObject, 700 );
  if ( cansee == true ) {
    m_lastKnownPos = m_pFollowObject->position();
    m_shouldFollow = true;
  }
  
  if ( m_shouldFollow == false ) {
    this->reset();
    this->m_counter = 5 + rand() % 10;
    return;
  }
  
  // If we're close enough to the leader-object. We stop.
  eng2d::Vec2D dirVec( m_lastKnownPos );
  dirVec -= m_pObject->position();
  if ( dirVec.length() < m_followDistance ) {
    // We close enough. Now we check the actual distance the object, not only
    // to the last known position. This makes it possible for us to go through
    // the teleporters
    if ( Utils::objectDistance( m_pObject, m_pFollowObject ) < 700 ) {
      this->reset();
      this->m_counter = 5 + rand() % 10;
      m_shouldFollow = false;
      return;
    }
  }
  
  // We're so far away from the leader-object that we must move towards him.
  int d = Utils::findTurningDir( m_pObject->position(),
                                 m_lastKnownPos,
                                 m_pObject->angle() );
  this->m_turn = 3 * d;
  this->m_forward = 1;
}



/** Makes the object to follow a route
 */
void NpcController::followRoute()
{
}



/** Stands still and shoots aliens when they're close enought
 */
void NpcController::fight()
{
  if ( m_aliensInSight == false ) {
    this->reset();
    m_counter = rand() % 10;
    m_pAlienToShoot = 0;
    return;
  }
  
  if ( m_pAlienToShoot == 0 ) {
    // We must find an alien to shoot.
    eng2d::Vec2D tmpPos( m_pObject->position() );
    float dist = 320 * 320;
    for ( int i=0; i < WarGlobals::pObjTable->objectList.size(); i++ ) {
      GameObject* pO = WarGlobals::pObjTable->objectList.at(i);
      if ( pO->state() == GameObject::STATE_LIVING ) {

        int oid = pO->objectType();
        bool ok = (oid == ObjectID::TYPE_CARNIVOREALIEN) ||
                  (oid == ObjectID::TYPE_SMALLWORMALIEN) ||
                  (oid == ObjectID::TYPE_WINGEDALIEN) ||
                  (oid == ObjectID::TYPE_PROCTORALIEN) ||
                  (oid == ObjectID::TYPE_MINIGUNALIEN) ||
                  (oid == ObjectID::TYPE_PREDATORALIEN) ||
                  (oid == ObjectID::TYPE_LIGHTBALLALIEN) ||
                  (oid == ObjectID::TYPE_FLAMERALIEN);
        eng2d::Vec2D dirVec( tmpPos );
        dirVec -= pO->position();
        if ( dirVec.length2() > dist ) {
          ok = false;
        }
        if ( ok == true ) {
          // This is ok. But is there an obstackle between us.
          if ( Utils::objectsSeeEachOther( m_pObject, pO, 320 ) == true ) {
            m_pAlienToShoot = pO;
          }
          return;
        }
      }
    }
    // We couldn't found anyone
    m_pAlienToShoot = 0;
    return;
  }


  this->reset();
  
  // Is our target already killed.
  if ( m_pAlienToShoot->state() == GameObject::STATE_KILLED ||
       m_pAlienToShoot->state() == GameObject::STATE_DYING ) {
    m_pAlienToShoot = 0;
    m_counter = rand() % 3;
    return;
  }
  
  // If the distance to the alien is too big we reject it and find a new
  // target.
  if ( Utils::objectDistance( m_pObject, m_pAlienToShoot ) > 320 ) {
    m_pAlienToShoot = 0;
    m_counter = rand() % 3;
    return;
  }
  
  // Aim the target and shoot when we're facing it.
  if ( Utils::isFacingObject( m_pObject, m_pAlienToShoot, 30 ) == true ) {
    // We're pointing him. Shoot.
    m_shoot = 1;
    m_counter = 5 + rand() % 10;
    return;
  }
  
  // We're not pointing him. We turn
  int d = Utils::findTurningDir( m_pObject->position(),
                                 m_pAlienToShoot->position(),
                                 m_pObject->angle() );
  m_turn = 2 * d;
  m_counter = 2;
}



/** Scans for aliens
 */
bool NpcController::scanForAliens()
{
  bool ret = false;
  float dist = 320 * 320;
  eng2d::Vec2D objPos( m_pObject->position() );
  
  for ( int i=0; i < WarGlobals::pObjTable->objectList.size(); i++ ) {
    GameObject* pO = WarGlobals::pObjTable->objectList.at(i);
    if ( pO->state() != GameObject::STATE_KILLED ) {
      ObjectID::Type oid = pO->objectType();
      bool alienObj = (oid == ObjectID::TYPE_CARNIVOREALIEN) ||
                      (oid == ObjectID::TYPE_SMALLWORMALIEN) ||
                      (oid == ObjectID::TYPE_WINGEDALIEN) ||
                      (oid == ObjectID::TYPE_PROCTORALIEN) ||
                      (oid == ObjectID::TYPE_MINIGUNALIEN) ||
                      (oid == ObjectID::TYPE_SENTRYGUN) ||
                      (oid == ObjectID::TYPE_LIGHTBALLALIEN) ||
                      (oid == ObjectID::TYPE_MEDIUMWORMALIEN) ||
                      (oid == ObjectID::TYPE_FLAMERALIEN);
                      
      if ( alienObj == true ) {
        eng2d::Vec2D dirVec( objPos );
        dirVec -= pO->position();
        if ( dirVec.length2() < dist ) {
          if ( Utils::objectsSeeEachOther( m_pObject, pO, 320 ) == true ) {
            return true;
          }
        }
      }
    }
  }
  return false;
}

} // end of namespace

/**
 * Version history
 * ===============
 * $Log: npccontroller.cpp,v $
 * Revision 1.2  2006/06/17 21:48:08  lamminsa
 * More supported aliens.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:43  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:15:36+02  lamminsa
 * Initial revision
 *
 */
 
